/* --- Copyright University of Sussex 1994.  All rights reserved. ---------
 > File:           C.unix/lib/lib/ptyfork.p
 > Purpose:        Fork a subprocess through pseudo-tty
 > Author:         Roger Evans, Dec 10 1986 (see revisions)
 > Documentation:  HELP PTYFORK
 > Related Files:  LIB VSH
 */

#_TERMIN_IF DEF POPC_COMPILING

/*  LIB PTYFORK         R.Evans November 1986

    UNSUPPORTED --- SEE DISCLAIMER IN HELP FILE

   This library lets you fork a process from poplog whose standard io
   is a pseudo tty not a pipe. It returns false (if it cannot) or a
   device (IO) for talking to the child process with and the name of the
   pty it is using. The tty is set to be the child's controlling tty. It
   does a double fork, so you don't have to worry about killing the child.
*/

uses ioctl_IO
uses ioctl_tchars
uses ioctl_ltchars
uses ioctl_sgttyb

/* some more ioctls not in these library files */
section;
global constant
        TIOCNOTTY = ioctl_IO(`t`,113),
        TIOCPKT = ioctl_IOW(`t`,112,4),
        TIOCREMOTE = ioctl_IOW(`t`,105,4);
endsection;

section $-library => ptyfork, get_pty_pair, pty_tchars, pty_ltchars;

/* -- locate and return a pseudo_tty pair -------------------------------- */

/*  Pseudo-tty masters (ptyp0 - ptypf) can only be opened by one process
    at a time so you have to look for the first one that will open.
    Having found it you can open the slave end (ttyp0 - ttypf).

    get_pty_pair returns master and slave devices (opened for io)
    or mishaps if it can't find one
*/

lconstant master = '/dev/ptyp0', slave = '/dev/ttyp0';

define get_pty_util(trygc);
    lvars i, j, d, trygc;

    /* trap mishap generated by sysopen (why can't it return false!) */
    define dlocal prmishap(s,l);
        lvars m s l;
        if isstring(l) then l else s endif -> m;
        if issubstring('CAN\'T OPEN FILE',1,m) then
            /* return false to caller - nb: hope nothing is on the stack! */
            false; exitfrom(sysopen);
        else
            sysprmishap(s,l);
        endif;
    enddefine;

    define lconstant try_pty;
        i -> master(10);
        j -> master(9);
        if sysopen(master,2,true) ->> d then
            d;
            i -> slave(10);
            j -> slave(9);
            copy(slave);
            exitfrom(get_pty_util);
        endif;
    enddefine;

    fast_for j from `p` to `r` do
        fast_for i from `0` to `9` do try_pty() endfor;
        fast_for i from `a` to `f` do try_pty() endfor;
    endfor;

    if trygc then
        /* see if there are garbage ones */
        sysgarbage();
        /* try again, but without further garbage collection */
        get_pty_util(false)
    else
        return(false);
    endif;
enddefine;

define global get_pty_pair;
    get_pty_util(true);
enddefine;

/* -- spawning other UNIX processes --------------------------------------- */


lconstant ioctl_null = inits(4), tty_mode = initshortvec(3);

;;; next two are accessed by LIB VED_VSH so must be global
global constant pty_tchars = inits(6);
global constant pty_ltchars = inits(6);

define lconstant sys_io_control_check(/* dev, cmd, arg */);
    unless sys_io_control(/* dev, cmd, arg */) then
        mishap(0, 'IOCTL FAILED');
    endunless;
enddefine;

define global ptyfork(file,args,env);
    lvars file args env master slave pid d;
    dlocal popdprecision = true;
    if get_pty_pair() ->> slave then
        -> master;
        /* fetch terminal modes of current popdevout */
        sys_io_control_check(popdevout,TIOCGETP,tty_mode);
        sys_io_control_check(popdevout,TIOCGETC,pty_tchars);
        sys_io_control_check(popdevout,TIOCGLTC,pty_ltchars);
        if sys_fork(true) ->> pid then
            /* clear packet and remote modes */
            sys_io_control_check(master,TIOCPKT,ioctl_null);
            sys_io_control_check(master,TIOCREMOTE,ioctl_null);
            master;
            slave;
            sys_wait(pid) -> (,);
        else
            sysclose(master);
            /* tidy up to make child pop system exit quietly */
            identfn -> popexit;
            false -> vedediting;
            [] -> vedbufferlist;
            /* throw away current devices */
            sysopen('/dev/null',2,false) ->> popdevin ->> poprawdevin
                    ->> poprawdevout ->> popdevout -> popdeverr;
            if sys_vfork(false) then
                sysexit();
            else
                /* clear out controlling tty for subprocess */
                sys_io_control_check(sysopen('/dev/tty',0,false), TIOCNOTTY);
                /* open new device (also becomes controlling tty) */
                sysopen(slave,2,false) -> popdevin;
                /* set terminal mode and special chars */
                tty_mode(3) &&~~ ECHO &&~~ CBREAK &&~~ RAW &&~~ CRMOD -> tty_mode(3);
                sys_io_control_check(popdevin,TIOCSETP,tty_mode);
                sys_io_control_check(popdevin,TIOCSETC,pty_tchars);
                sys_io_control_check(popdevin,TIOCSLTC,pty_ltchars);
                /* now copy to other devices */
                popdevin ->> popdevout -> popdeverr;
                sysexecute(file,args,env);
                /* just in case something goes wrong */
                sysclose(slave);
                sysexit();
            endif;
        endif;
    else
        false;
    endif;
enddefine;

endsection;


/* --- Revision History ---------------------------------------------------
--- John Gibson, Apr 21 1994
        Uses new sys_fork etc
--- Robert John Duncan, Jun 29 1992
        Added checks that the ioctls actually work and moved to C.unix
--- Robert John Duncan, Jun  1 1992
        Removed redundant calls to -sys_io_control-
--- Aaron Sloman, Oct 24 1988
        Put in public, non-autoloadable library
 */
