/* --- Copyright University of Sussex 1995. All rights reserved. ----------
 > File:            C.all/lib/objectclass/src/trees.p
 > Purpose:         Objectclass file
 > Author:          Steve Knight, HP Labs, 1992-1993 (see revisions)
 > Documentation:   HELP OBJECTCLASS
 > Related Files:
 */
;;; -- Trees ----------------------------------------------------------------
;;; A declarative representation of the method dispatch table, suitable
;;; for (efficient) optimisation.
;;;
;;; Trees are easily the most complex data structure in the library.  The
;;; tree has a two stage life-cycle.  It is initially built, for a particular
;;; method, with the nextTree field bound to a list of {class tree} pairs.
;;;
;;; The second stage occurs when the tree is instantiated.  This process
;;; alters the nextTree field into a list of Branch([keys,...] [trees,...]).
;;; [Also, note that the sets of keys are mutually exclusive.]
;;;
;;; The second stage is useful for generating the code -- since the keys
;;; can be used for direct "==" comparision and the sequence of trees
;;; represent alternatives generated by competition between classes for
;;; keys.  The difference between the two stages can be summarised as
;;; that between compilation time and run-time.
;;;

compile_mode :pop11 +strict;

section $-objectclass;

defclass Tree [writeable] {
	actionTree,     ;;; This is the method part.
	nextTree,       ;;; This is the next set of arcs, two different reps.
	defaultTree,    ;;; This is the default case.
	/* hashTree : pint ;;; The hash code for the tree */
};

constant procedure newTree = consTree;


/*
	EXPERIMENT INVESTIGATING HASHING FUNCTIONS FOR BETTER PERFORMANCE
lvars nhash = 0;
define newTree( x, y, z ); lvars x, y, z;
	nhash fi_+ 1 -> nhash;
	if nhash > 1e8 then
		0 -> nhash
	endif;
	consTree( x, y, z, nhash )
enddefine;

hashTree -> class_hash( Tree_key );
*/

defclass Branch {
	keysBranch,     ;;; The list of alternative keys.
	treesBranch,    ;;; The list of alternative trees.
};

constant procedure newBranch = consBranch;

;;; t represents a tree that cannot fail (i.e. will always apply).
define cantFailTree( t ); lvars t, d;
	t.actionTree or
	(t.defaultTree ->> d).isTree and d.cantFailTree
enddefine;

;;; One of the members of L is bound to apply.
define oneCantFail( L ); lvars L;
	L.null.not and ( L.hd.cantFailTree or L.tl.oneCantFail )
enddefine;

define simpleTree( T ); lvars T;
	nextTree( T ) == [] and not( defaultTree( T ) )
enddefine;

;;; T is a duff tree -- it has no action, default, or nextTree.
;;; It should be eliminated, of course.
define nullTree( T ); lvars T;
	not( actionTree( T ) ) and simpleTree( T )
enddefine;

endsection;


/* --- Revision History ---------------------------------------------------
--- Robert John Duncan, May 26 1995
		Added [writeable] attribute to Tree class definition
 */
